<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\Payment;
use App\Models\Quotation;
use Illuminate\Http\Request;

class PaymentController extends Controller
{
    public function pay($id)
    {
        // Fetch the quotation using booking_id
        $quotation = Quotation::where('booking_id', $id)
            ->with(['booking', 'booking.user', 'booking.address'])
            ->firstOrFail(); // Ensure the quotation exists

        // Fetch all quotation items related to this booking
        $items = Quotation::where('booking_id', $quotation->booking_id)->get();

        return view('payment.index', compact('quotation', 'items'));
    }

    public function show($id)
    {
        // Fetch the payments using booking_id
        $payment = Payment::where('booking_id', $id)
            ->with(['booking.subService', 'booking.quotation'])
            ->firstOrFail();

        return view('payment.show', compact('payment'));
    }

    public function index()
    {
        // Check if the user is authenticated
        if (!auth()->check()) {
            // Redirect to login or show an error
            return redirect()->route('login')->with('error', 'Sila log masuk untuk mengakses halaman ini.');
        }

        $role = auth()->user()->role;

        if (!in_array($role, ['admin', 'owner'])) {
            abort(403, 'Unauthorized action.');
        }

        $layout = $role === 'admin' ? 'layouts.baseAdmin' : 'layouts.baseSP';

        $bookings = Booking::whereHas('payments')
            ->with(['user', 'service', 'subService', 'payments'])
            ->get();

        foreach ($bookings as $booking) {
            $booking->latest_payment = $booking->payments->first() ?? null;
        }

        //dev url
        // $toyyibpayUrl = 'https://dev.toyyibpay.com/index.php/bill';

        //production url
        $toyyibpayUrl = 'https://toyyibpay.com/index.php/bill';
        


        return view('Admin.payment.index', compact('layout', 'bookings', 'toyyibpayUrl'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|string', // Ensure status is provided and is a string
        ]);

        try {
            $booking = Booking::findOrFail($id);
            $booking->status = $request->input('status');
            $booking->save();

            return redirect()->back()->with('success', 'Status berjaya dikemas kini.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Terdapat ralat semasa mengemas kini status.');
        }
    }



    public function verify()
    {
        return view('Admin.payment.verify'); //payment.verify
    }
}
