<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\Payment;
use App\Models\Service;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class HomeController extends Controller
{
    // public function __construct()
    // {
    //    $this->middleware('auth'); 
    // }

    public function index()
    {
        return view('homepage');
    }

    public function userHome()
    {
        return view('homepage'); // Customer dashboard
    }

    public function adminHome()
    {
        // Fetch bookings with related user and service data
        $bookings = Booking::with(['user', 'service', 'subService'])
            ->get();

        // Fetching data for "Jumlah Pengguna" (User Count) - Only users with the 'user' role
        $userCount = User::where('role', 'user')->count();

        // Fetching data for "Jumlah Tempahan" (Booking Count)
        $bookingCount = Booking::count();

        // Fetching data for "Jumlah Bayaran" (Total Payment)
        $totalPayment = Payment::sum('amount'); // Assuming Payment table holds the payment data

        return view('Admin.homepage', compact('userCount', 'bookingCount', 'totalPayment', 'bookings'));
    }

    public function ownerHome()
    {
        // Fetching the count of bookings for each service
        $services = Service::all();  // Assuming you have a Service model that relates to each service

        // Create an array to hold the counts for each service
        $serviceCounts = [];

        foreach ($services as $service) {
            // Count how many times each service has been booked
            $serviceCounts[] = Booking::where('service_id', $service->id)->count();
        }

        // Fetching data for "Jumlah Pengguna" (User Count) - Only users with the 'user' role
        $userCount = User::where('role', 'user')->count();

        // Fetching data for "Jumlah Tempahan" (Booking Count)
        $bookingCount = Booking::count();

        // Fetching data for "Jumlah Bayaran" (Total Payment)
        $totalPayment = Payment::sum('amount'); // Assuming Payment table holds the payment data

        return view('Owner.homepage', compact('userCount', 'bookingCount', 'totalPayment', 'serviceCounts', 'services'));
    }




    // Profile Controller
    public function profile()
    {
        $user = Auth::user();

        // Default image placeholder
        $user->profile_image = $user->profile_image ?: 'default-profile.png';

        return view('profile', compact('user'));
    }

    // Update Profile Image
    public function updateProfileImage(Request $request)
    {
        // Get the authenticated user
        $user = User::find(auth()->id());

        // Validate the uploaded image
        $request->validate([
            'profile_image' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ]);

        // Handle profile image upload
        if ($request->hasFile('profile_image')) {
            // Remove the old profile image if it exists
            if ($user->profile_image && file_exists(public_path('uploads/profile_images/' . $user->profile_image))) {
                unlink(public_path('uploads/profile_images/' . $user->profile_image));
            }

            // Save the new profile image
            $image = $request->file('profile_image');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('uploads/profile_images'), $imageName);

            // Update the user's profile image in the database
            $user->profile_image = $imageName;
        } else {
            // If no image is uploaded, ensure a default image is used
            if (!$user->profile_image) {
                $user->profile_image = 'default_profile_image.jpg';
            }
        }

        // Save the user record
        $user->save();

        return redirect()->back()->with('success', 'Imej profil berjaya dikemaskini.');
    }

    // Update Personal Information
    public function updateInfo(Request $request)
    {
        // Fetch the authenticated user
        $userId = auth()->id();
        if (!$userId) {
            return redirect()->back()->withErrors(['error' => 'User is not authenticated']);
        }

        $user = User::find($userId);
        if (!$user) {
            return redirect()->back()->withErrors(['error' => 'User not found']);
        }

        // Validate the input
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phoneNum' => 'nullable|string|max:15',
            'gender' => 'nullable|string|in:Lelaki,Perempuan,Lain-lain',
            'dob' => 'nullable|date|before:today',
        ]);

        // Update the user's details
        $updated = $user->update($request->only(['name', 'email', 'phoneNum', 'gender', 'dob']));

        if ($updated) {
            return redirect()->back()->with('success', 'Maklumat peribadi berjaya dikemaskini.');
        }

        return redirect()->back()->withErrors(['error' => 'Failed to update personal information.']);
    }

    // Change Password
    public function changePassword(Request $request)
    {
        $user = User::find(auth()->id());

        // Validate the request
        $request->validate([
            'current_password' => 'required',
            'new_password' => [
                'required',
                'confirmed',
                'min:6', // At least 6 characters
                'regex:/[a-z]/', // At least one lowercase letter
                'regex:/[A-Z]/', // At least one uppercase letter
                'regex:/[0-9]/', // At least one number
                'regex:/[@$!%*#?&]/', // At least one special character
            ],
        ]);

        // Check if the current password matches
        if (!Hash::check($request->current_password, $user->password)) {
            return redirect()->back()->with('error', 'Kata laluan semasa adalah tidak betul.');
        }

        // Update the user's password
        $user->update(['password' => Hash::make($request->new_password)]);

        return redirect()->back()->with('success', 'Kata laluan berjaya dikemaskini.');
    }

    // Delete Account
    public function deleteAccount(Request $request)
    {
        $user = User::find(auth()->id());

        // Delete user's profile image if it exists
        if ($user->profile_image && file_exists(public_path('uploads/profile_images/' . $user->profile_image))) {
            unlink(public_path('uploads/profile_images/' . $user->profile_image));
        }

        // Delete the user's account
        $user->delete();

        // Redirect to the home page with a success message
        return redirect('/')->with('success', 'Akaun anda telah berjaya dipadam.');
    }
}
