<?php

namespace App\Http\Controllers;

use App\Models\Address;
use App\Models\Booking;
use App\Models\Service;
use App\Models\SubService;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class BookingController extends Controller
{

    //Customer

    public function create()
    {
        $services = Service::with('subServices')->get(); // Retrieve services with sub-services

        return view('booking.create', compact('services')); // Pass data to the view

    }

    public function getAvailableSlots(Request $request)
    {
        $date = $request->input('date');

        if (!$date || !strtotime($date)) {
            return response()->json(['error' => 'Invalid or missing date parameter'], 400);
        }

        $timeSlots = [
            '09:00',
            '10:00',
            '11:00',
            '12:00',
            '13:00',
            '14:00',
            '15:00',
            '16:00',
            '17:00'
        ];

        $bookedSlots = Booking::where('date', $date)->pluck('time')->toArray();

        $formattedSlots = collect($timeSlots)->map(function ($time) use ($bookedSlots) {
            return [
                'time' => $time,
                'is_available' => !in_array($time, $bookedSlots),
            ];
        });

        return response()->json(['timeSlots' => $formattedSlots]);
    }

    public function store(Request $request)
    {
        // Validate incoming data
        $validated = $request->validate([
            'service' => 'required|exists:services,id',
            'sub_service' => 'required|exists:sub_services,id',
            'parts' => 'required|string|in:Ya,Tidak',
            'unit' => 'integer|min:1',
            'date' => 'required|date',
            'time' => 'required|date_format:H:i',
            'alt_date' => 'nullable|date',
            'alt_time' => 'nullable|date_format:H:i',
            'proof_damage' => 'nullable|file|mimes:jpg,png,jpeg,gif,svg,mp4,mov,avi|max:20480', // Allow images and videos (max 20MB)
            'address_line1' => 'required|string',
            'address_line2' => 'nullable|string',
            'city' => 'required|string',
            'postcode' => 'required|string',
            'remark' => 'nullable|string',
        ]);

        // Custom validation for conflicting slots on the same date
        if ($validated['alt_date'] === $validated['date'] && $validated['time'] === $validated['alt_time']) {
            return back()->withErrors(['alt_time' => 'Slot masa alternatif tidak boleh sama dengan slot masa pilihan pada tarikh yang sama!']);
        }

        // Handle file upload if present
        $proofDamage = null;
        if ($request->hasFile('proof_damage')) {
            $proofDamage = $request->file('proof_damage')->store('proof_damage', 'public');
        }

        // Fetch the estimate_price of the selected sub-service
        $subService = SubService::findOrFail($request->input('sub_service'));
        $estimatePrice = $subService->estimate_price;

        // Create Address record first
        $address = Address::create([
            'address_line1' => $request->input('address_line1'),
            'address_line2' => $request->input('address_line2'),
            'city' => $request->input('city'),
            'postcode' => $request->input('postcode'),
        ]);

        // Create the booking and associate it with the address
        $booking = new Booking();
        $booking->user_id = auth()->id(); // Associate booking with the logged-in user
        $booking->service_id = $request->input('service');
        $booking->sub_service_id = $request->input('sub_service');
        $booking->parts = $request->input('parts');
        $booking->unit = $request->input('unit');
        $booking->date = $request->input('date');
        $booking->time = $request->input('time');
        $booking->alt_date = $request->input('alt_date');
        $booking->alt_time = $request->input('alt_time');
        $booking->remark = $request->input('remark');
        $booking->proof_damage = $proofDamage;
        $booking->address_id = $address->id; // Store the foreign key to the Address
        $booking->status = 'Hantar'; // Set the status to "Hantar"

        $booking->save();

        // Redirect or return success
        return redirect()->route('booking.index')->with('success', 'Tempahan berjaya dicipta!');
    }


    public function index()
    {
        // Fetch current bookings (statuses excluding 'Selesai')
        $currentBookings = Booking::with('service', 'subService')
            ->where('user_id', auth()->id())
            ->whereNotIn('status', ['Selesai']) // Exclude 'Selesai'
            ->get();

        // Fetch historical bookings (status: 'Selesai')
        $historyBookings = Booking::with('service', 'subService')
            ->where('user_id', auth()->id())
            ->where('status', 'Selesai')
            ->get();

        // Pass both to the view
        return view('booking.index', compact('currentBookings', 'historyBookings'));
    }



    public function show($id)
    {
        // Fetch the booking data along with the associated address, service, and sub_service
        $booking = Booking::with('address', 'service', 'subService')->findOrFail($id);

        // Return the show view with the booking data
        return view('booking.show', compact('booking'));
    }


    public function progress($id)
    {
        $booking = Booking::with(['service', 'subService', 'quotation'])->find($id);

        if (!$booking) {
            abort(404, 'Booking not found');
        }

        // Define the workflow and active steps
        $workflow = [
            'Hantar' => [
                'icon' => 'bi-send',
                'title' => 'Tempahan Dihantar',
                'description' => 'Tempahan anda telah dihantar dan sedang menunggu kelulusan daripada pihak kami.',
                'status' => 'info',
            ],
            'Lulus' => [
                'icon' => 'bi-check-circle',
                'title' => 'Tempahan Diluluskan',
                'description' => 'Tempahan anda telah diluluskan. Kami sedang menyediakan sebut harga untuk perkhidmatan yang diminta.',
                'status' => 'info',
                'link' => route('booking.show', ['id' => $booking->id]),
                'link_text' => 'Lihat Tempahan',
            ],
            'Sebut Harga' => [
                'icon' => 'bi-receipt',
                'title' => 'Sebut Harga Tersedia',
                'description' => 'Sebut harga anda telah tersedia. Sila semak dan sahkan dalam tempoh 7 hari daripada tarikh dikeluarkan.',
                'status' => 'primary',
                'link' => route('quotation.show', ['id' => $booking->id]),
                'link_text' => 'Lihat Sebut Harga',
            ],
            'Tolak' => [
                'icon' => 'bi-exclamation-circle',
                'title' => 'Sebut Harga Ditolak',
                'description' => 'Anda telah memilih untuk menolak sebut harga ini. Hubungi kami untuk bantuan lanjut jika diperlukan.',
                'status' => 'warning',
            ],
            'Menunggu Sebut Harga' => [
                'icon' => 'bi-clock-history',
                'title' => 'Menunggu Sebut Harga Baharu',
                'description' => 'Permintaan untuk sebut harga baharu sedang diproses oleh pihak kami.',
                'status' => 'info',
            ],
            'Sebut Harga Baru' => [
                'icon' => 'bi-clipboard-check',
                'title' => 'Sebut Harga Baharu Tersedia',
                'description' => 'Sebut harga baharu telah dikeluarkan. Sila semak dan sahkan dalam tempoh 7 hari dari tarikh dikeluarkan.',
                'status' => 'info',
                'link' => route('quotation.show', ['id' => $booking->id]),
                'link_text' => 'Lihat Sebut Harga',
            ],
            'Belum Bayar' => [
                'icon' => 'bi-wallet',
                'title' => 'Menunggu Pembayaran',
                'description' => 'Anda telah menerima sebut harga. Sila buat pembayaran untuk memulakan kerja baikpulih.',
                'status' => 'warning',
                'link' => $booking->status === 'Belum Bayar' ? route('payment.pay', ['id' => $booking->id]) : null,
                'link_text' => $booking->status === 'Belum Bayar' ? 'Buat Pembayaran' : null,
            ],
            'Pembayaran Berjaya' => [
                'icon' => 'bi-receipt',
                'title' => 'Pembayaran Berjaya',
                'description' => 'Pembayaran anda telah berjaya. Sila tunggu kelulusan daripada tukang baiki sebelum kerja dimulakan.',                
                'status' => 'success',
                'link' => route('payment.display', ['id' => $booking->id]),
                'link_text' => 'Lihat Resit Pembayaran',
            ],
            'Pembayaran Tidak Selesai' => [
                'icon' => 'bi-wallet2',
                'title' => 'Pembayaran Belum Dilakukan',
                'description' => 'Anda belum membuat pembayaran untuk tempahan ini. Sila membuat pembayaran segera untuk meneruskan pembayaran.',
                'status' => 'info',
                'link' => route('payment.display', ['id' => $booking->id]),
                'link_text' => 'Buat Pembayaran',
            ],
            'Pembayaran Gagal' => [
                'icon' => 'bi-x-circle',
                'title' => 'Pembayaran Gagal',
                'description' => 'Transaksi FPX anda tidak berjaya. Sila cuba semula atau hubungi pihak bank untuk bantuan lanjut.',
                'status' => 'danger',
                'link' => route('payment.display', ['id' => $booking->id]),
                'link_text' => 'Buat Pembayaran',
            ],
            'Kerja Dalam Proses' => [
                'icon' => 'bi-tools',
                'title' => 'Kerja Sedang Dilaksanakan',
                'description' => 'Kerja sedang dijalankan. Kami akan mengemas kini status setelah selesai.',
                'status' => 'primary',
            ],
            'Selesai' => [
                'icon' => 'bi-check-circle',
                'title' => 'Tempahan Selesai',
                'description' => 'Kerja yang ditempah telah selesai dengan jayanya. Terima kasih kerana menggunakan perkhidmatan kami!',
                'status' => 'success',
                'link' => route('invoice.show', ['id' => $booking->id]),
                'link_text' => 'Lihat Invois',
            ],
        ];

        // Conditional logic to unset steps based on the current booking status
        if ($booking->status === 'Belum Bayar') {
            unset($workflow['Tolak'], $workflow['Menunggu Sebut Harga'], $workflow['Sebut Harga Baru']);
        }

        if ($booking->status === 'Pembayaran Berjaya') {
            unset($workflow['Pembayaran Tidak Selesai'], $workflow['Pembayaran Gagal']);
        }

        if ($booking->status === 'Pembayaran Tidak Selesai') {
            unset($workflow['Pembayaran Berjaya'], $workflow['Pembayaran Gagal']);
        }

        if ($booking->status === 'Pembayaran Gagal') {
            unset($workflow['Pembayaran Berjaya'], $workflow['Pembayaran Tidak Selesai']);
        }

        if ($booking->status === 'Kerja Dalam Proses') {
            unset($workflow['Pembayaran Tidak Selesai'], $workflow['Pembayaran Gagal']);
        }

        if ($booking->status === 'Selesai') {
            unset($workflow['Pembayaran Tidak Selesai'], $workflow['Pembayaran Gagal']);
        }

        // Filter the workflow to display only up to the current status
        $filteredWorkflow = [];
        foreach ($workflow as $key => $step) {
            $filteredWorkflow[] = $step;
            if ($key === $booking->status) {
                break;
            }
        }

        // Reverse the workflow to display the latest step at the top
        $filteredWorkflow = array_reverse($filteredWorkflow);

        return view('booking.progress', compact('booking', 'filteredWorkflow'));
    }



    // Show the form for editing an existing booking
    public function edit($id)
    {
        // Fetch the booking data by ID
        $booking = Booking::findOrFail($id);

        // Fetch the address details associated with the booking
        $address = $booking->address;

        // Ensure 'time' and 'alt_time' are formatted to 'H:i' (only time, no date)
        $booking->time = $booking->time ? $booking->time->format('H:i') : null;
        $booking->alt_time = $booking->alt_time ? $booking->alt_time->format('H:i') : null;

        // Define available time slots in 'H:i' format
        $timeSlots = [
            '09:00',
            '10:00',
            '11:00',
            '12:00',
            '13:00',
            '14:00',
            '15:00',
            '16:00',
            '17:00'
        ];

        // Get the list of all services and sub-services
        $services = Service::all();
        $sub_services = SubService::all();

        // Pass the booking, address, services, sub_services, and timeSlots to the view
        return view('booking.edit', compact('booking', 'address', 'services', 'sub_services', 'timeSlots'));
    }


    // Handle the update logic
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'service' => 'required|exists:services,id',
            'sub_service' => 'required|exists:sub_services,id',
            'parts' => 'required|string|in:Ya,Tidak',
            'unit' => 'required|integer|min:1',
            'date' => 'required|date',
            'time' => 'required|date_format:H:i',
            'alt_date' => 'nullable|date',
            'alt_time' => 'nullable|date_format:H:i|required_if:alt_date,!=,null',
            'proof_damage' => 'nullable|image|mimes:jpg,png,jpeg,gif,svg|max:2048',
            'address_line1' => 'required|string',
            'address_line2' => 'nullable|string',
            'city' => 'required|string',
            'postcode' => 'required|string',
            'remark' => 'nullable|string',
        ]);

        // Check if time and alt_time overlap on the same date
        if (
            $request->date === $request->alt_date &&
            $request->time === $request->alt_time
        ) {
            return redirect()->back()->withErrors([
                'alt_time' => 'Masa dan masa alternatif tidak boleh sama jika tarikh adalah sama.',
            ])->withInput();
        }

        // Fetch the existing booking
        $booking = Booking::findOrFail($id);

        // Handle file upload for proof of damage (if any)
        if ($request->hasFile('proof_damage')) {
            $validated['proof_damage'] = $request->file('proof_damage')->store('proof_damage', 'public');
        }

        // Update the Address model (if the address has changed)
        $address = Address::findOrFail($booking->address_id);
        $address->update([
            'address_line1' => $request->input('address_line1'),
            'address_line2' => $request->input('address_line2'),
            'city' => $request->input('city'),
            'postcode' => $request->input('postcode'),
        ]);

        // Update the booking
        $booking->update($validated);

        return redirect()->route('booking.index')->with('success', 'Tempahan dikemaskini dengan berjaya!');
    }

    public function destroy(Booking $booking)
    {
        try {
            $booking->delete();
            return redirect()->route('booking.index')->with('success', 'Tempahan berjaya dipadam.');
        } catch (\Exception $e) {
            return redirect()->route('booking.index')->with('error', 'Terdapat ralat semasa memadam tempahan.');
        }
    }



    //Admin

    public function list()
    {
        // Retrieve bookings with status "Hantar" and their related data
        $bookings = Booking::with(['user', 'service', 'subService'])
            ->where('status', 'Hantar')
            ->get();

        // Pass bookings to the view
        return view('Admin.booking.index', compact('bookings'));
    }

    public function showViewPage($id)
    {
        $booking = Booking::with(['user', 'service', 'subService', 'address'])->findOrFail($id);

        // Parse the time field using Carbon if it's not null
        if ($booking->time) {
            $booking->time = Carbon::parse($booking->time);
        }

        // Return the view with booking data
        return view('Admin.booking.show', compact('booking'));
    }


    public function showApprovalPage($id)
    {
        $booking = Booking::with(['user', 'service', 'subService', 'address'])->findOrFail($id);

        // Return the view with booking data
        return view('Admin.booking.approve', compact('booking'));
    }

    public function approve(Request $request, $id)
    {
        $validated = $request->validate([
            'approved_slot' => 'required|string', // Format: "date|time"
        ]);

        // Split the approved slot into date and time
        [$approvedDate, $approvedTime] = explode('|', $validated['approved_slot']);

        // Fetch the booking record
        $booking = Booking::findOrFail($id);

        // Update the approved date and time
        $booking->approved_date = $approvedDate;
        $booking->approved_time = $approvedTime;
        $booking->status = 'Lulus'; // Update status to "Approved"

        $booking->save();

        // Redirect or return a success response
        return redirect()->route('quotation.create', ['id' => $id])->with('success', 'Tempahan telah diluluskan.');
    }

    public function reject(Request $request, $id)
    {
        $request->validate([
            'reject_reason' => 'required|string|max:255',
        ]);

        $booking = Booking::findOrFail($id);
        $booking->status = 'Gagal';
        $booking->reject_reason = $request->reject_reason;
        $booking->save();

        return redirect()->route('booking.list')->with('success', 'Tempahan telah ditolak.');
    }

    public function custList()
    {
        // Retrieve bookings with status "Hantar" and their related data
        $bookings = Booking::with(['user', 'service', 'subService'])
            ->get();

        // Pass bookings to the view
        return view('Owner.booking.index', compact('bookings'));
    }

    public function showCust($id)
    {
        $booking = Booking::with(['user', 'service', 'subService', 'address'])->findOrFail($id);

        // Return the view with booking data
        return view('Owner.booking.show', compact('booking'));
    }
}
