<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Sale;
use App\Models\Harvest;
use App\Models\Price;
use App\Models\Plantation;
use Illuminate\Http\Request;

class SaleController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $sales = Sale::with([
            'harvest' => function ($query) {
                $query->withTrashed()->with([
                    'plantation' => function ($query) {
                        $query->withTrashed()->with([
                            'plantDetail' => function ($query) {
                                $query->withTrashed();
                            }
                        ]);
                    }
                ]);
            },
            'customer'
        ])->get();



        // Aggregate total_sale by year
        $yearlySales = [];
        foreach ($sales as $sale) {
            $year = date('Y', strtotime($sale->created_at)); // Extract the year from the date
            if (!isset($yearlySales[$year])) {
                $yearlySales[$year] = 0;
            }
            $yearlySales[$year] += $sale->total_sale; // Sum total_sale for each year
        }

        // Prepare data for the graph
        $graphData = [];
        foreach ($yearlySales as $year => $totalSale) {
            $graphData[] = [
                'year' => $year, // Use the year as the x-axis value
                'value' => $totalSale // Use the aggregated total_sale as the y-axis value
            ];
        }

        // Aggregate grade_a_sale, grade_b_sale, and grade_c_sale
        $gradeSales = [
            'Grade A' => $sales->sum('grade_a_sale'),
            'Grade B' => $sales->sum('grade_b_sale'),
            'Grade C' => $sales->sum('grade_c_sale')
        ];

        // Prepare data for the second graph (grade sales)
        $gradeSalesData = [];
        foreach ($gradeSales as $grade => $totalSale) {
            $gradeSalesData[] = [
                'grade' => $grade, // Use the grade as the category
                'value' => $totalSale // Use the aggregated sale as the value
            ];
        }


        // Calculate total sales for each grade
        $totalSales = $sales->sum('total_sale');
        $totalSalesGradeA = $sales->sum('total_sale_grade_a');
        $totalSalesGradeB = $sales->sum('total_sale_grade_b');
        $totalSalesGradeC = $sales->sum('total_sale_grade_c');

        return view('sales.index', compact(
            'sales',
            'graphData',
            'gradeSalesData',
            'totalSales',
            'totalSalesGradeA',
            'totalSalesGradeB',
            'totalSalesGradeC'
        ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // $harvests = Harvest::with('plantation')
        //     ->where(function ($query) {
        //         $query->where('grade_a_available', '>', 0)
        //             ->orWhere('grade_b_available', '>', 0)
        //             ->orWhere('grade_c_available', '>', 0);
        //     })
        //     ->get();
        $harvests = Harvest::with([
            'plantation' => function ($query) {
                $query->withTrashed()->with([
                    'plantDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            }
        ])
            ->where(function ($query) {
                $query->where('grade_a_available', '>', 0)
                    ->orWhere('grade_b_available', '>', 0)
                    ->orWhere('grade_c_available', '>', 0);
            })
            ->withTrashed() // Include soft-deleted harvests
            ->get();


        $customers = Customer::all();
        $price = Price::latest()->first();

        // $price = Price::latest();

        return view('sales.create', compact('harvests', 'customers', 'price'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreSaleRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'harvest_id' => 'required|exists:harvests,id',
            'price_id' => 'required|exists:prices,id',
            'customer_id' => 'required|exists:customers,id',
            'grade_a_sale' => 'required|integer',
            'grade_b_sale' => 'required|integer',
            'grade_c_sale' => 'required|integer',
            'total_sale_grade_a' => 'required|numeric',
            'total_sale_grade_b' => 'required|numeric',
            'total_sale_grade_c' => 'required|numeric',
            'total_sale' => 'required|numeric',
        ]);

        $harvest = Harvest::findOrFail($request->harvest_id);
        // Update the available quantities
        $harvest->grade_a_available -= $request->grade_a_sale; // Subtract grade A sale from available
        $harvest->grade_b_available -= $request->grade_b_sale; // Subtract grade B sale from available
        $harvest->grade_c_available -= $request->grade_c_sale; // Subtract grade C sale from available
        $harvest->status = 'Jual';
        $harvest->save();


        $sale = new Sale();
        $sale->harvest_id = $request->harvest_id;
        $sale->price_id = $request->price_id;
        $sale->customer_id = $request->customer_id;
        $sale->grade_a_sale = $request->grade_a_sale;
        $sale->grade_b_sale = $request->grade_b_sale;
        $sale->grade_c_sale = $request->grade_c_sale;
        $sale->total_sale_grade_a = $request->total_sale_grade_a;
        $sale->total_sale_grade_b = $request->total_sale_grade_b;
        $sale->total_sale_grade_c = $request->total_sale_grade_c;
        $sale->total_sale = $request->total_sale;

        $sale->save();
        return redirect()->route('sales.index')->with('success', 'Maklumat Pembeli Berjaya Disimpan.');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sale  $sales
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {

        $sale = Sale::with([
            'harvest' => function ($query) {
                $query->withTrashed()->with([
                    'plantation' => function ($query) {
                        $query->withTrashed()->with([
                            'plantDetail' => function ($query) {
                                $query->withTrashed();
                            }
                        ]);
                    }
                ]);
            },
            'customer',
            'price'
        ])->findOrFail($id);
        $chartData = [
            [
                'grade' => 'Grade A',
                'value' => $sale->grade_a_sale // Sale quantity for Grade A
            ],
            [
                'grade' => 'Grade B',
                'value' => $sale->grade_b_sale // Sale quantity for Grade B
            ],
            [
                'grade' => 'Grade C',
                'value' => $sale->grade_c_sale // Sale quantity for Grade C
            ]
        ];

        return view('sales.show', compact('sale', 'chartData'));
    }

    public function receipt($id)
    {
        $sales = Sale::with([
            'harvest' => function ($query) {
                $query->withTrashed()->with([
                    'plantation' => function ($query) {
                        $query->withTrashed()->with([
                            'plantDetail' => function ($query) {
                                $query->withTrashed();
                            }
                        ]);
                    }
                ]);
            },
            'customer',
            'price'
        ])->findOrFail($id);
        // $sales = Sale::findOrFail($id);
        return view('sales.receipt', compact('sales'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sale  $sales
     * @return \Illuminate\Http\Response
     */
    public function edit()
    {
        $sales = Sale::All();
        return view('sales.edit', compact('sales'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\UpdateSaleRequest  $request
     * @param  \App\Models\Sale  $sales
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Sale $sales)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sale  $sales
     * @return \Illuminate\Http\Response
     */
    public function destroy(Sale $sales)
    {
        //
    }
}
